%% Supplementary software for the article:
%
% "Photon count estimation in single-molecule localization microscopy"
% Rasmus . Thorsen, Christiaan N. Hulleman, Sjoerd Stallinga and Bernd Rieger
%
% This .m file reproduces the experimental part of Figure 1B. 
% - (Example 3) The photometric ratio 
% 1) Load experimental aberration-corrected through-focus PSF stack
% 2) Estimate photon count by Gaussian (MLE-)fit
% 3) Estimate photon count by TRABI
% 4) Compute photometric ratio (Gaussian fit over TRABI value)
%
% Mean and standard deviation are calculated for two different beads
% measured three times each, i.e, 6 configurations in total.
%
% - figure 1
% Estimated aberration coefficients for all tertiary Zernike modes
% (root-mean-sqaure values).
%
% - figure 2
% The photometric ratio: P = Gaussian fit / TRABI 

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands  

%% Example 3 - initialize
clearvars
close all
clc

help example3

%% Photometric ratio

vec_spot = [1 3]; % two beads
vec_data = [8 9 10]; % three measurements for each bead

zerncoeff = zeros(6,21);
P = zeros(17,6);
rem_focus = zeros(1,6);

for nspot = 1:length(vec_spot)
    for ndata = 1:length(vec_data)
        
        addpath('exampledata') % add psf scripts
        
        % load experimental aberration corrected through-focus PSF stacks
        load(['bead45nm_' num2str(vec_data(ndata)) '_21x21x21_teritary_spot' num2str(vec_spot(nspot))],'allspots','thetastore','parameters')
        Nph = parameters.signalphotoncount; % Estimated photon count (vectorial fit)
        Nbg = parameters.backgroundphotoncount; % Estimated background per pixel (vectorial fit)
        rmpath('exampledata') % remove psf scripts
        
        %% Estimate number of photons
        
        %%%% Gaussian fit
        addpath('matlabfun/gaussian fitter') % add gaussian fitt scripts
        ROI = 7; % region of interest for gaussian fitting [px]
        theta_gaussian = get_gaussfitter(allspots,ROI,parameters); % output: theta_gaussian: estimated (x,y,z,sigma,Nph,Nbg)
        rmpath('matlabfun/gaussian fitter') % remove gaussian fit scripts
        
        %%%% TRABI
        addpath('matlabfun/trabi') % add trabi scripts
        radius = 214.5*1.86/parameters.pixelsize; % aperture radius in trabi [px] - FWHM_PSF of set parameters is 214.5 nm
        theta_trabi = get_trabi(allspots,radius,Nbg,parameters); % output: theta_trabi: estimated (Nph)
        rmpath('matlabfun/trabi') % remove trabi scripts
        
        % adjust out of focus frames
        idx_focus = round(parameters.zemit/parameters.pixelsize);
        rem_focus(ndata+(nspot-1)*length(vec_data)) = -parameters.zemit+parameters.pixelsize*idx_focus;
        Nph_gaussian = theta_gaussian((3:end-2)+idx_focus,4);
        Nph_trabi = theta_trabi((3:end-2)+idx_focus);
        
        % Photometric ratio
        P(:,ndata+(nspot-1)*length(vec_data)) = Nph_gaussian./Nph_trabi*100;
        
        %% Estimated zernike coefficients
        numzers = parameters.numparams-5;
        thetafinal = squeeze(thetastore(:,:,end));
        zerncoeff(ndata+(nspot-1)*length(vec_data),:) = thetafinal(4:3+numzers,1);
        
    end
end

% compute mean and std variation of photometric ratio
Pmean = mean(P,2);
Pstd = std(P.');

% compute image (x,y,z)-coordinates
ImageSizex = parameters.Mx*parameters.pixelsize/2;
ImageSizey = parameters.My*parameters.pixelsize/2;
ximag = -ImageSizex+parameters.pixelsize/2:parameters.pixelsize:ImageSizex;
yimag = -ImageSizey+parameters.pixelsize/2:parameters.pixelsize:ImageSizey;
[XImage,YImage] = meshgrid(ximag,yimag);
zrange = -parameters.zrange;
Mz = parameters.Mz;
ImageSizez = zrange;
DzImage = 2*ImageSizez/Mz;
ZImage = (-ImageSizez+DzImage/2:DzImage:ImageSizez);
ZImage = ZImage(3:end-2)+mean(rem_focus);

%% Plot

%%% Photon count and photometric ratio
scrsz = [1 1 1366 768];
figure
errorbar(ZImage,Pmean,Pstd)
ylabel('Photometric ratio [%]')
xlabel('Axial position')
title('Gaussian fit over TRABI')
axis square
xlim([-600 600])

% Estimated Zernike coefficients 
figure
set(gcf,'Position',[.5*scrsz(4) 0.7*scrsz(4) 1.0*scrsz(4) 0.3*scrsz(4)]);
errorbar(1:numzers,mean(zerncoeff,1),std(zerncoeff,1))

ax = gca; % edit for older versions 
allxticks = 1:numzers;
allxticklabels = cell(numzers,1);
% xticks(allxticks)
ax.XTick = allxticks;
orders = parameters.aberrations(:,1:2);
for jzer = 1:numzers
    allxticklabels{jzer} = strcat(num2str(orders(jzer,1)),',',num2str(orders(jzer,2)));
end
% xticklabels(allxticklabels)
ax.XTickLabel = allxticklabels;
% xtickangle(45)
ax.XTickLabelRotation = 45;
ylabel('Aberration coefficient [m\lambda]');
xlabel('Zernike mode');
ylim([-20 20])
xlim([0 numzers+1])